'''
pyGnGAnalysis -- Analysing Go/NoGo Experiment Results

@author:     Pasieka Manuel , manuel.pasieka@csf.ac.at

@copyright:  2015 BioComp/CSF. All rights reserved.

@license:    license

@contact:    manuel.pasieka@csf.ac.at
@deffield    updated: Updated
'''

import sys
import logging
import time
from argparse import ArgumentParser
from argparse import RawDescriptionHelpFormatter
import configparser
import os.path

__all__ = []
PROJECT_NAME = 'pyGnGAnalysis'
__version__ = 0.1
__date__ = '2015-09-17'
__updated__ = '2015-09-17'

__logfile__ = 'pyGnGAnalysis.log'
__logfilelevel__ = logging.DEBUG

def helperIsFile(parser, filename):
    fName = os.path.abspath(filename)
    if os.path.isfile(fName):
        return fName
    else:
        parser.error('%s is no file' % (fName))


def createArgumentParser():
    """
    Create an Argument Parser
    """
    program_version = "v%s" % __version__
    program_build_date = str(__updated__)
    program_version_message = '%%(prog)s %s (%s)' % (program_version, program_build_date)
    program_shortdesc = __import__('__main__').__doc__.split("\n")[1]
    program_license = '''%s

      Created by Pasieka Manuel on %s.
      Copyright 2015 BioComp. All rights reserved.

      Distributed on an "AS IS" basis without warranties
      or conditions of any kind, either express or implied.

    ''' % (program_shortdesc, str(__date__))

    # Setup argument parser
    parser = ArgumentParser(prog=PROJECT_NAME,
                            description=program_license,
                            formatter_class=RawDescriptionHelpFormatter)
    parser.add_argument('movementData',
                        help="Path to file containing the movement data",
                        type=lambda x: helperIsFile(parser, x))
    parser.add_argument('-c', '--config', default="config.ini",
                        help="Config file")
    parser.add_argument('-L', '--logfile', default=__logfile__,
                        help="Log file")
    parser.add_argument("-v", "--verbosity", action='count', default=0,
                        help="increase output verbosity. Use -vv, -vvv for more")
    parser.add_argument('--version', action='version', version=program_version_message)

    return parser


def setupLogging(logfile, logfilelevel, logconsolelevel):
    """
    Define python.logging to file and console
    """
    #Normal settings
    #logformat = '[%(levelname)s] %(module)s.%(funcName)s : %(message)s'

    #Debugging settings
    logformat = '[%(levelname)s] \'%(filename)s +%(lineno)d\' %(funcName)s : %(message)s'

    #Setup logging to file
    logging.basicConfig(level=logfilelevel, format=logformat,
                        filename=logfile, filemode="w")

    #Add logging to the console
    console = logging.StreamHandler()
    console.setLevel(logconsolelevel)
    console.setFormatter(logging.Formatter(logformat))
    logging.getLogger('').addHandler(console)


def readConfigFile(fileName, tagName):
    """
    Read python config file using ConfigParser
    Will read all options under the region "tagName" and if possible convert a
    value into float.
    """
    logging.debug(format('Reading config file [%s] ...' % fileName))
    Config = configparser.ConfigParser()
    Config.optionxform = str  # Keeps options to be all set to lowercases
    Config.read_file(open(fileName))
    options = Config.options(tagName)

    parameters = dict()
    for option in options:
        try:
            parameters[option] = Config.get(tagName, option)
            try:
                #If possible convert to number
                parameters[option] = float(parameters[option])
            except ValueError:
                pass
        except KeyError:
            logging.error(format("Exception on config file option [%s]!" % option))
            parameters[option] = None

    return parameters


def main(argv=None):  # IGNORE:C0111
    """
    Transition Analysis
    """
    try:
        parser = createArgumentParser()
        try:
            args = parser.parse_args()
        except SystemExit:
            logging.error("Parsing arguments failed!")
            raise

        if args.verbosity >= 2:
            consoleloglevel = logging.DEBUG
        elif args.verbosity >= 1:
            consoleloglevel = logging.INFO
        else:
            consoleloglevel = logging.WARN

        setupLogging(args.logfile, __logfilelevel__, consoleloglevel)

        logging.info(format('Starting %s v%s on the %s...'
                            % (PROJECT_NAME, __version__, time.ctime())))

        logging.info(format("Reading config file [%s]" % (args.config)))
        parameters = readConfigFile(args.config, PROJECT_NAME)

        #XXX YOUR CODE XXX

        logging.info(format('Finished %s' % PROJECT_NAME))
        print("Finished %s" % PROJECT_NAME)

        return 0
    except Exception as e:
        logging.error('Execution failed %s', e)
        return 2


if __name__ == "__main__":
    sys.exit(main(sys.argv))
